#ifdef __KERNEL__
#ifdef MODULE

#include <linux/config.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/kdev_t.h>
#include <linux/proc_fs.h>
/* NOTE:
 * RH8.0 by default turn of CONFIG_DEVFS_FS kernel support,
 * that meant devfs_register() will always return NULL ...
 */
#include <linux/devfs_fs_kernel.h>
#include <linux/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/pci.h>
#include "pcitg_kernel.h"
#include "pcitg_ioctl.h"

static struct file_operations pcitg_fops;

MODULE_AUTHOR("Kuang-chun Cheng <kccheng@openate.com>");
MODULE_DESCRIPTION("PCI Target I/O Linux driver");
MODULE_LICENSE("GPL");

unsigned short debug = 0;
MODULE_PARM(debug, "h");
MODULE_PARM_DESC(debug, "debug level (0-0xFFFF)");

short major = 0;
MODULE_PARM(major, "h");

static devfs_handle_t devfs_handle;

static pcitg_board_t boards[PCITG_MAX_BOARD];

static char banner[] __initdata =
	KERN_INFO "PCI Target I/O Linux driver\n"
	          "Copyright (C) 2003, OpenATE, INC.\n"
	          "author: Kuang-Chun Cheng <kccheng@openate.com>\n";

static int num_board_found = 0;

static void
pcitg_irq_handler(int irq, void *dev_id, struct pt_regs *regs)
{
        pcitg_board_t *pb = NULL;
	struct task_struct *p = NULL;
	struct siginfo info;
	unsigned char old;

	//
	// dev_id is used as minor number, or index to boards[]
	//
	pb = &boards[*((int *)dev_id)];


#if 0
// I believe this should not be necessary ...

	if (!pb->pci) { pb = NULL; continue; }
	if (pb->irq != irq) { pb = NULL; continue; }
	if ((inb(PCITG_IRQ_CTL(pb->base)) & 0x10)) { pb = NULL; continue; }
#endif

	// IRQ ACK
	old = inb(PCITG_IRQ_CTL(pb->base));
	outb(0xB0 | (old & 0xF), PCITG_IRQ_CTL(pb->base));

	pb->irqmask = (inb(PCITG_MBOX_IN(pb->base)) >> 4) & 0x0F &
		      (inb(PCITG_MBOX_IN(pb->base)) & 0x0F);

	p = find_task_by_pid(pb->pid);
	if (p) {

		info.si_signo = pb->signo; // this MUST be set
		info.si_code = SI_QUEUE;
		info.si_value.sival_int = (*((int *)dev_id)) |
					((pb->irqmask << 16) & 0xFFFF0000);

		if (pb->signo) kill_proc_info(pb->signo, &info, pb->pid);
	}

	// Re-enable IRQ
	old = inb(PCITG_IRQ_CTL(pb->base));
	outb(0xC0 | (old & 0xF), PCITG_IRQ_CTL(pb->base));
	return;
}


static int
pcitg_read_proc_debug(char *buf, char **start, off_t offset, int count,
	int *eof, void *data)
{
	int len;
	len = sprintf(buf, "%hd\n", debug);
	*eof = 1;
	return len;
}


static int
pcitg_write_proc_debug (struct file *file, const char *buf,
	unsigned long count, void *data)
{
	char in[2];
	if (count < 0 || count > 2) {
		printk(KERN_WARNING "invalid input count = %ld\n", count);
		return -EINVAL;
	}
        if (copy_from_user (&in, buf, count)) return -EFAULT;

	if (in[0] == '1') {
		printk(KERN_INFO "turn on debugging mode\n");
		debug = 1;
	} else if (in[0] == '0') {
		printk(KERN_INFO "turn off debugging mode\n");
		debug = 0;
	} else {
		printk(KERN_INFO "debugging mode = %d, unchange\n", debug);
	}

	return count;
}


static int
pcitg_open(struct inode *inode, struct file *file)
{
	int minor;
	pcitg_board_t *pb;	// pointer to board

	minor = MINOR(inode->i_rdev);
	if (debug) printk(KERN_INFO "pcitg_open %d/%d ...\n", major, minor);

	if (!boards[minor].pci) return -ENODEV;
	pb = &boards[minor];

	if (!pb->used_count) {
		if (!request_region(
			pci_resource_start(pb->pci, 0),
			pci_resource_len(pb->pci, 0), "pcitg")) {
			goto base_err;
                }
                pb->base = pci_resource_start(pb->pci, 0);
		printk("bar0: 0x%08lX ~ 0x%08lX (0x%08lX)\n",
                        pci_resource_start(pb->pci, 0), pci_resource_end(pb->pci, 0),
                        pci_resource_len(pb->pci, 0));

		pb->irq = pb->pci->irq;
		if (request_irq(pb->irq, pcitg_irq_handler,
			SA_SHIRQ, "pcitg", &minor) < 0) {
			goto irq_err;
		}
		file->private_data = pb;
		if (debug) printk(KERN_INFO "irq: %d\n", pb->irq);

		outb(0xAA, PCITG_IRQ_CTL(pb->base));
		outb(0x80, PCITG_CHIP0_PORT_CTL(pb->base));
		outb(0x00, PCITG_CHIP0_PORT_A(pb->base));
		outb(0x00, PCITG_CHIP0_PORT_B(pb->base));
		outb(0x00, PCITG_CHIP0_PORT_C(pb->base));
		outb(0x80, PCITG_CHIP1_PORT_CTL(pb->base));
		outb(0x00, PCITG_CHIP1_PORT_A(pb->base));
		outb(0x00, PCITG_CHIP1_PORT_B(pb->base));
		outb(0x00, PCITG_CHIP1_PORT_B(pb->base));
		outb(0x00, PCITG_MBOX_OUT(pb->base));
		outb(0x00, PCITG_MBOX_IN(pb->base));

		pb->used_count++;
	} else {
		printk("board[%d] in used\n", minor);
		return -EBUSY;
	}

	MOD_INC_USE_COUNT;
	if (debug) printk(KERN_INFO "    MOD_IN_USE = %d\n", MOD_IN_USE);

        return 0;

	if (pb->irq) {
                free_irq(pb->irq, &minor);
		pb->irq = 0;
	}
irq_err:

        release_region(
                pci_resource_start(pb->pci, 0),
                pci_resource_len(pb->pci, 0));
base_err:
	return -1;
}

static int
pcitg_ioctl (struct inode *inode, struct file *file,
unsigned int cmd, unsigned long arg)
{
	pcitg_board_t *pb = NULL;
	if (debug) printk(KERN_INFO "pcitg_ioctl ...\n");
        pb = (pcitg_board_t *) file->private_data;
        if (!pb->pci) return -ENODEV;

	switch (cmd) {
	case PCITG_IOC_IN: {
		iodata_t data;
                if (debug > 100) printk(KERN_INFO "        PCITG_IOC_IN\n");

		if (copy_from_user(&data, (iodata_t *) arg, sizeof(data))) return -EFAULT;

		if (data.chip == PCITG_GROUP0) {
			switch(data.addr) {
			case PCITG_PORT_A:
				data.value = inb(PCITG_CHIP0_PORT_A(pb->base)) & 0xFF;
				break;
			case PCITG_PORT_B:
				data.value = inb(PCITG_CHIP0_PORT_B(pb->base)) & 0xFF;
				break;
			case PCITG_PORT_C:
				data.value = inb(PCITG_CHIP0_PORT_C(pb->base)) & 0xFF;
				break;
			case PCITG_PORT_CTL:
				data.value = inb(PCITG_CHIP0_PORT_CTL(pb->base)) & 0xFF;
				break;
			case PCITG_PORT_ALL: {
				unsigned char aa, bb, cc, CC;
				CC = inb(PCITG_CHIP0_PORT_CTL(pb->base)) & 0xFF;
				aa = inb(PCITG_CHIP0_PORT_A(pb->base)) & 0xFF;
				bb = inb(PCITG_CHIP0_PORT_B(pb->base)) & 0xFF;
				cc = inb(PCITG_CHIP0_PORT_C(pb->base)) & 0xFF;
				data.value = CC + (aa << 8) + (bb << 16) + (cc << 24);
				break; }
			default:
				return -ENOTTY;
			}
		} else if (data.chip == PCITG_GROUP1) {
			switch(data.addr) {
			case PCITG_PORT_A:
				data.value = inb(PCITG_CHIP1_PORT_A(pb->base)) & 0xFF;
				break;
			case PCITG_PORT_B:
				data.value = inb(PCITG_CHIP1_PORT_B(pb->base)) & 0xFF;
				break;
			case PCITG_PORT_C:
				data.value = inb(PCITG_CHIP1_PORT_C(pb->base)) & 0xFF;
				break;
			case PCITG_PORT_CTL:
				data.value = inb(PCITG_CHIP1_PORT_CTL(pb->base)) & 0xFF;
				break;
			case PCITG_PORT_ALL: {
				unsigned char aa, bb, cc, CC;
				CC = inb(PCITG_CHIP1_PORT_CTL(pb->base)) & 0xFF;
				aa = inb(PCITG_CHIP1_PORT_A(pb->base)) & 0xFF;
				bb = inb(PCITG_CHIP1_PORT_B(pb->base)) & 0xFF;
				cc = inb(PCITG_CHIP1_PORT_C(pb->base)) & 0xFF;
				data.value = CC + (aa << 8) + (bb << 16) + (cc << 24);
				break; }
			default:
				return -ENOTTY;
			}
		} else {
			return -ENOTTY;
		}

		if (copy_to_user((iodata_t *)arg, &data, sizeof(data))) return -EFAULT;


                break; }
	case PCITG_IOC_OUT: {
		iodata_t data;
                if (debug > 100) printk(KERN_INFO "        PCITG_IOC_OUT\n");

		if (copy_from_user(&data, (iodata_t *) arg, sizeof(data))) return -EFAULT;

		if (data.chip == PCITG_GROUP0) {
			switch(data.addr) {
			case PCITG_PORT_A:
				outb(data.value & 0xFF, PCITG_CHIP0_PORT_A(pb->base));
				break;
			case PCITG_PORT_B:
				outb(data.value & 0xFF, PCITG_CHIP0_PORT_B(pb->base));
				break;
			case PCITG_PORT_C:
				outb(data.value & 0xFF, PCITG_CHIP0_PORT_C(pb->base));
				break;
			case PCITG_PORT_CTL:
				outb(data.value & 0xFF, PCITG_CHIP0_PORT_CTL(pb->base));
				break;
			case PCITG_PORT_ALL: {
				unsigned char aa, bb, cc, CC;

				CC = (data.value & 0xFF);
				outb(CC, PCITG_CHIP0_PORT_CTL(pb->base));
				aa = ((data.value & 0xFF00) >> 8) & 0xFF;
				outb(aa, PCITG_CHIP0_PORT_A(pb->base));
				bb = ((data.value & 0xFF0000) >> 16) & 0xFF;
				outb(bb, PCITG_CHIP0_PORT_B(pb->base));
				cc = ((data.value & 0xFF000000) >> 24) & 0xFF;
				outb(cc, PCITG_CHIP0_PORT_C(pb->base));

				break; }
			default:
				return -ENOTTY;
			}

		} else if (data.chip == PCITG_GROUP1) {
			switch(data.addr) {
			case PCITG_PORT_A:
				outb(data.value & 0xFF, PCITG_CHIP1_PORT_A(pb->base));
				break;
			case PCITG_PORT_B:
				outb(data.value & 0xFF, PCITG_CHIP1_PORT_B(pb->base));
				break;
			case PCITG_PORT_C:
				outb(data.value & 0xFF, PCITG_CHIP1_PORT_C(pb->base));
				break;
			case PCITG_PORT_CTL:
				outb(data.value & 0xFF, PCITG_CHIP1_PORT_CTL(pb->base));
				break;
			case PCITG_PORT_ALL: {
				unsigned char aa, bb, cc, CC;

				CC = (data.value & 0xFF);
				outb(CC, PCITG_CHIP1_PORT_CTL(pb->base));
				aa = ((data.value & 0xFF00) >> 8) & 0xFF;
				outb(aa, PCITG_CHIP1_PORT_A(pb->base));
				bb = ((data.value & 0xFF0000) >> 16) & 0xFF;
				outb(bb, PCITG_CHIP1_PORT_B(pb->base));
				cc = ((data.value & 0xFF000000) >> 24) & 0xFF;
				outb(cc, PCITG_CHIP1_PORT_C(pb->base));

				break; }
			default:
				return -ENOTTY;
			}
		} else {
			return -ENOTTY;
		}

                break; }

	case PCITG_IOC_SET_IRQMASK: {
		int data, old;
                if (debug > 100) printk(KERN_INFO "        PCITG_IOC_SET_IRQMASK:\n");
		if (copy_from_user(&data, (int *) arg, sizeof(data)))
			return -EFAULT;

		old = inb(PCITG_IRQ_CTL(pb->base));
		outb((data & 0xF) | (old & 0xF), PCITG_IRQ_CTL(pb->base));

                break; }

	case PCITG_IOC_GET_IRQMASK: {
		int data;
                if (debug > 100) printk(KERN_INFO "        PCITG_IOC_GET_IRQMASK:\n");

		data = inb(PCITG_IRQ_CTL(pb->base));
		data = data & 0xF;

		if (copy_to_user((int *)arg, &data, sizeof(data)))
			return -EFAULT;
                break; }
		
	case PCITG_IOC_MBOX_IN: {
		unsigned char data;
                if (debug > 100) printk(KERN_INFO "        PCITG_IOC_MBOX_IN\n");

		data = inb(PCITG_MBOX_IN(pb->base));
		if (copy_to_user((unsigned char *)arg, &data, sizeof(data)))
			return -EFAULT;

                break; }
	case PCITG_IOC_MBOX_OUT: {
		unsigned char data;
                if (debug > 100) printk(KERN_INFO "        PCITG_IOC_MBOX_OUT\n");

		if (copy_from_user(&data, (unsigned char *) arg, sizeof(data)))
			return -EFAULT;
		outb(data, PCITG_MBOX_OUT(pb->base));

                break; }

	case PCITG_IOC_IRQ_CONNECT: {
		irqdata_t data;
		unsigned char old;

                printk(KERN_INFO "        PCITG_IOC_IRQ_CONNECT\n");
		if (copy_from_user(&data, (irqdata_t *) arg, sizeof(data)))
			return -EFAULT;
		pb->signo = data.signo;
		pb->pid   = data.pid;

		if (pb->signo) {
			// enable PCITG IRQ
			printk("Enable IRQ, write 0xC0\n");
			old = inb(PCITG_IRQ_CTL(pb->base));
			outb(0xC0 | (old & 0xF), PCITG_IRQ_CTL(pb->base));
			printk("Enable IRQ, get 0x%X\n",
				inb(PCITG_IRQ_CTL(pb->base)));
		} else {
			// disable PCITG IRQ
			old = inb(PCITG_IRQ_CTL(pb->base));
			outb(0x80 | (old & 0xF), PCITG_IRQ_CTL(pb->base));
		}

                break; }

	case PCITG_IOC_SLOTNAME: {
                if (debug > 100) printk(KERN_INFO "        PCITG_IOC_SLOTNAME\n");
		if (copy_to_user((char *) arg, pb->pci->slot_name, 8)) {
                        return -EFAULT;
                }
                break; }
	default:
		// return -EINVAL;	// this make sense and ... OK
		return -ENOTTY;		// however, POSIX said we should return
					// -ENOTTY.
	}
	return 0;
}

static int
pcitg_release(struct inode *inode, struct file *file)
{
	int minor;
	pcitg_board_t *pb;
        pb = (pcitg_board_t *) file->private_data;
	if (!pb->pci) return -ENODEV;

	minor = MINOR(inode->i_rdev);

	if (pb->used_count == 1) {
		if (pb->irq) {
			free_irq(pb->irq, &minor);
			pb->irq = 0;
		}

		outb(0xAA, PCITG_IRQ_CTL(pb->base));
		outb(0x00, PCITG_CHIP0_PORT_CTL(pb->base));
		outb(0x00, PCITG_CHIP0_PORT_A(pb->base));
		outb(0x00, PCITG_CHIP0_PORT_B(pb->base));
		outb(0x00, PCITG_CHIP0_PORT_C(pb->base));
		outb(0x00, PCITG_CHIP1_PORT_CTL(pb->base));
		outb(0x00, PCITG_CHIP1_PORT_A(pb->base));
		outb(0x00, PCITG_CHIP1_PORT_B(pb->base));
		outb(0x00, PCITG_CHIP1_PORT_C(pb->base));
		outb(0x00, PCITG_MBOX_OUT(pb->base));
		outb(0x00, PCITG_MBOX_IN(pb->base));

		release_region(
			pci_resource_start(pb->pci, 0),
			pci_resource_len(pb->pci, 0));

	}
	pb->used_count--;
        MOD_DEC_USE_COUNT;
	if (debug) printk(KERN_INFO "pcitg_release ...\n");
	if (debug) printk(KERN_INFO "    MOD_IN_USE = %d\n", MOD_IN_USE);

        return 0;
}


static int __init pcitg_init(void)
{
	int i;
	struct pci_dev *ppci = NULL;
        struct proc_dir_entry *ent;
	char path[64];

	if (debug) printk(KERN_INFO "pcitg_init ...\n");

	for (i=0; i<PCITG_MAX_BOARD; ++i) {
		boards[i].pci		= NULL;
		boards[i].base		= 0;
		boards[i].irq		= 0;
		boards[i].irqmask	= 0;
		boards[i].used_count	= 0;
		boards[i].signo		= 0;
		boards[i].pid		= 0;
	}

	// search all PCI devices here
	for (;;) {
		ppci = pci_find_device(PCITG_VENDOR_ID, PCITG_DEVICE_ID, ppci);
		if (num_board_found > PCITG_MAX_BOARD) {
			// NOTE, order of checking is important
			num_board_found--;
			break;
		} else if (!ppci) {
			break;
		} else {
			if (ppci &&
			    ppci->subsystem_vendor == PCITG_SUBSYSTEM_VENDOR_ID &&
			    ppci->subsystem_device == PCITG_SUBSYSTEM_ID) {
				boards[num_board_found].pci = ppci;
				printk(KERN_INFO "    found PCITG %d\n", num_board_found);
				++num_board_found;
			}
		}
	}
	if (!num_board_found) {
		printk(KERN_ERR "PCITG not found\n");
		return -ENODEV;
	} else {
		printk(KERN_INFO "TOTAL found PCITG %d\n", num_board_found);
	}

	if ((major = devfs_register_chrdev(0, "pcitg", &pcitg_fops)) < 0) {
		printk(KERN_ERR "pcitg: unable to register driver\n");
		goto reg_err;
	} else {
		printk(KERN_INFO "pcitg: major %d\n", major);
	}

	devfs_handle = devfs_register(NULL, "pcitg",
		DEVFS_FL_DEFAULT, major, 0,
		S_IFCHR | S_IRUSR | S_IWUSR, &pcitg_fops, NULL);

	printk(banner);

	if (!(ent = proc_mkdir("pcitg", NULL))) {
		printk(KERN_ERR "can't create dir /proc/pcitg\n");
		goto proc_pcitg_err; }

	// for (i=0; i<PCITG_MAX_BOARD; ++i) {
	for (i=0; i<num_board_found; ++i) {
		sprintf(path, "pcitg/board%d", i);

		if (!(ent = proc_mknod(path, 0664|S_IFCHR,
			NULL, mk_kdev(major,i)))) {
			printk(KERN_ERR "can't create dir %s\n", path);
			goto proc_pcitg_path_err; }
	}

	if (!(ent = create_proc_entry("pcitg/debug", S_IFREG | S_IRUGO | S_IWUGO, 0))) {
                printk(KERN_ERR "can't create dir /proc/pcitg/debug\n");
                goto proc_pcitg_debug_err; }
	ent->read_proc = pcitg_read_proc_debug;
	ent->write_proc = pcitg_write_proc_debug;

	return 0;

        remove_proc_entry("pcitg/debug", NULL);
proc_pcitg_debug_err:

	// for (i=0; i<PCITG_MAX_BOARD; ++i) {
	for (i=0; i<num_board_found; ++i) {
		sprintf(path, "pcitg/board%02d", i);
        	remove_proc_entry(path, NULL);
	}
proc_pcitg_path_err:

        remove_proc_entry("pcitg", NULL);
proc_pcitg_err:

	devfs_unregister_chrdev(major, "pcitg");
	devfs_unregister(devfs_handle);
reg_err:
	return -ENODEV;

}
module_init(pcitg_init);

static void __exit pcitg_exit(void)
{
	int i;
	char path[64];

	if (debug) printk(KERN_INFO "pcitg_exit ...\n");
        remove_proc_entry("pcitg/debug", NULL);
	for (i=0; i<num_board_found; ++i) {
		sprintf(path, "pcitg/board%02d", i);
        	remove_proc_entry(path, NULL);
	}
        remove_proc_entry("pcitg", NULL);

	for (i=0; i<PCITG_MAX_BOARD; ++i) {
		boards[i].pci = NULL;
	}

	devfs_unregister_chrdev(major, "pcitg");
	devfs_unregister(devfs_handle);
	return;
}
module_exit(pcitg_exit);


/*
 check <linux/fs.h>:

 NOTE
 read, write, poll, fsync, readv, writev can be called
 without the big kernel lock held in all filesystems.

struct file_operations {
        struct module *owner;
        loff_t (*llseek) (struct file *, loff_t, int);
        ssize_t (*read) (struct file *, char *, size_t, loff_t *);
        ssize_t (*write) (struct file *, const char *, size_t, loff_t *);
        int (*readdir) (struct file *, void *, filldir_t);
        unsigned int (*poll) (struct file *, struct poll_table_struct *);
        int (*ioctl) (struct inode *, struct file *,
		unsigned int, unsigned long);
        int (*mmap) (struct file *, struct vm_area_struct *);
        int (*open) (struct inode *, struct file *);
        int (*flush) (struct file *);
        int (*release) (struct inode *, struct file *);
        int (*fsync) (struct file *, struct dentry *, int datasync);
        int (*fasync) (int, struct file *, int);
        int (*lock) (struct file *, int, struct file_lock *);
        ssize_t (*readv) (struct file *,
		const struct iovec *, unsigned long, loff_t *);
        ssize_t (*writev) (struct file *,
		const struct iovec *, unsigned long, loff_t *);
        ssize_t (*sendpage) (struct file *,
		struct page *, int, size_t, loff_t *, int);
        unsigned long (*get_unmapped_area)(struct file *,
		unsigned long, unsigned long, unsigned long, unsigned long);
};

*/

static struct file_operations pcitg_fops = {
	open:		pcitg_open,
	ioctl:		pcitg_ioctl,
	release:	pcitg_release,
};


#endif /* MODULE */
#endif /* __KERNEL__ */
