#include <pcitg.h>
#include <pcitg_ioctl.h>

static pcitg_t *boards[32] = {
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL};

static void pcitg_handler(int signo, siginfo_t *info, void *data)
{
	int id;
	int vector;

	/*
	 * IRQ Vector or IRQ Source:
	 *
	 * 0: External Interrupt
	 * 1: Chip0: Interrupt Request B
	 * 2: Chip0: Interrupt Request A
	 * 4: Chip1: Interrupt Request B
	 * 8: Chip1: Interrupt Request A
	 *
	 */

	id     = info->si_int & 0x0000FFFF;
	vector = ((info->si_int & 0xFFFF0000) >> 16) & 0x0000FFFF;
	if (id < 0 || id >= 32) {
		printf("INTERNAL ERROR\n");
		return;
	}
	if (boards[id]->handler)
		return (boards[id]->handler)(boards[id], vector);
	return;
	
}

pcitg_t *pcitg_open(const int id)
{
	char path[128];
	pcitg_t *board = NULL;

	if (id < 0 || id >= 32)
		goto error;

	if (boards[id]) {
		// used
		goto error;
	}

	board = malloc(sizeof(struct pcitg));
	if (!board) {
		goto error;
	}

	snprintf(path, 128, "/proc/pcitg/board%d", id);

	board->fd = open(path, O_RDWR);
	if (board->fd < 0) {
		goto error;
	}

	board->id          = id;
	board->handler     = NULL;

	ioctl(board->fd, PCITG_IOC_SLOTNAME, board->slot);

	boards[id] = board;
	return board;
error:
	if (board) { free(board); board = NULL; }
	return PCITG_NULL;
}

void pcitg_close(pcitg_t *board)
{
	if (!board) return;
	if (board->handler) { board->handler = NULL; }
	if (boards[board->id]) { boards[board->id] = NULL; }
	if (board->id) { board->id = 0; }
	if (board->fd >= 0) close(board->fd);
	if (board) { free(board); board = NULL; }
}

char *pcitg_slot_name(pcitg_t *board)
{
	if (!board) return PCITG_NULL;
	return board->slot;
}

int pcitg_in(pcitg_t *board, int chip, int ports, unsigned long *value)
{
	iodata_t data;
	if (!board || !value) return PCITG_ERROR;
	if (chip != PCITG_CHIP0 && chip != PCITG_CHIP1) return PCITG_ERROR;

	data.chip  = chip;

	switch (ports) {
	case PCITG_PORT_A:   data.addr = PCITG_PORT_A;   break;
	case PCITG_PORT_B:   data.addr = PCITG_PORT_B;   break;
	case PCITG_PORT_C:   data.addr = PCITG_PORT_C;   break;
	case PCITG_PORT_CTL: data.addr = PCITG_PORT_CTL; break;
	case PCITG_PORT_ALL: data.addr = PCITG_PORT_ALL; break;
	default: return PCITG_ERROR;
	}

	if (ioctl(board->fd, PCITG_IOC_IN, &data) < 0)
		return PCITG_ERROR;

	switch (ports) {
	case PCITG_PORT_A:
	case PCITG_PORT_B:
	case PCITG_PORT_C:
	case PCITG_PORT_CTL: *value = data.value & 0xFF; break;
	case PCITG_PORT_ALL: *value = data.value;        break;
	default: return PCITG_ERROR;
	}

	return PCITG_OK;
}

int pcitg_out(pcitg_t *board, int chip, int ports, unsigned long value)
{
	iodata_t data;
	if (!board) return PCITG_ERROR;
	if (chip != PCITG_CHIP0 && chip != PCITG_CHIP1) return PCITG_ERROR;

	data.chip  = chip;
	data.value = value;

	switch (ports) {
	case PCITG_PORT_A:   data.addr = PCITG_PORT_A;   break;
	case PCITG_PORT_B:   data.addr = PCITG_PORT_B;   break;
	case PCITG_PORT_C:   data.addr = PCITG_PORT_C;	 break;
	case PCITG_PORT_CTL:
		data.addr = PCITG_PORT_CTL;
		board->ctl[chip] = value & 0xFF;
		break;
	case PCITG_PORT_ALL:
		data.addr = PCITG_PORT_ALL;
		board->ctl[chip] = value & 0xFF;
		break;
	default: return PCITG_ERROR;
	}

	if (ioctl(board->fd, PCITG_IOC_OUT, &data) < 0)
		return PCITG_ERROR;
	return PCITG_OK;
}

int pcitg_set_irqmask(pcitg_t *board, int value)
{
	if (ioctl(board->fd, PCITG_IOC_SET_IRQMASK, &value) < 0)
		return PCITG_ERROR;

	return PCITG_OK;
}

int pcitg_get_irqmask(pcitg_t *board, int *value)
{
	if (ioctl(board->fd, PCITG_IOC_GET_IRQMASK, value) < 0)
		return PCITG_ERROR;

	return PCITG_OK;
}

int pcitg_mbox_in(pcitg_t *board, unsigned char *value)
{
	if (!board || !value) return PCITG_ERROR;

	if (ioctl(board->fd, PCITG_IOC_MBOX_IN, value) < 0)
		return PCITG_ERROR;

	return PCITG_OK;
}

int pcitg_mbox_out(pcitg_t *board, unsigned char value)
{
	if (!board) return PCITG_ERROR;

	if (ioctl(board->fd, PCITG_IOC_MBOX_OUT, &value) < 0)
		return PCITG_ERROR;

	return PCITG_OK;
}

int pcitg_connect(pcitg_t *board, void (*sig_handler) (pcitg_t *, int))
{
	irqdata_t data;
	if (!board) return PCITG_ERROR;

	board->sa.sa_sigaction = pcitg_handler;
	sigemptyset(&board->sa.sa_mask);
	board->sa.sa_flags = SA_SIGINFO;
	if (sigaction(SIG_PCITG, &board->sa, NULL) < 0) {
		return PCITG_ERROR;
	}
	printf("connect signo %d to pid %d\n", SIG_PCITG, getpid());

	if (sig_handler) {
		data.signo = SIG_PCITG;
		data.pid   = getpid();

		if (ioctl(board->fd, PCITG_IOC_IRQ_CONNECT, &data) < 0) {
			return PCITG_ERROR;
		}
		board->handler = sig_handler;
	} else {
		data.signo = 0;	// side effect will disable IRQ
		data.pid   = getpid();

		if (ioctl(board->fd, PCITG_IOC_IRQ_CONNECT, &data) < 0) {
			return PCITG_ERROR;
		}
		board->handler = NULL;
	}
	return PCITG_OK;
}

